<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        $invoices = Invoice::with('payments')->latest()->paginate(10);
        $recentPayments = Invoice::with('payments')->where('status', 'paid')->latest()->get();
        $pendingInvoices = Invoice::with('payments')->where('status', 'pending')->latest()->get();
        return view('dashboard', compact('invoices', 'recentPayments', 'pendingInvoices'));
    }

    public function reinitializeLink(Invoice $invoice)
    {
        // Reset invoice status to pending and generate new token
        $invoice->update([
            'status' => 'pending',
            'token' => \Illuminate\Support\Str::random(32),
        ]);

        return redirect()->back()->with('success', 'Payment link reinitialized successfully!');
    }

    public function export()
    {
        $invoices = Invoice::with('payments')->get();

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('dashboard.export-pdf', compact('invoices'));

        $filename = 'invoices_export_' . now()->format('Y-m-d_H-i-s') . '.pdf';

        return $pdf->download($filename);
    }

    public function analytics(Request $request)
    {
        $invoices = Invoice::all();

        // Analytics data
        $analytics = [
            'total_invoices' => $invoices->count(),
            'paid_invoices' => $invoices->where('status', 'paid')->count(),
            'pending_invoices' => $invoices->where('status', 'pending')->count(),
            'cancelled_invoices' => $invoices->where('status', 'cancelled')->count(),
            'total_amount' => $invoices->sum('amount'),
            'paid_amount' => $invoices->where('status', 'paid')->sum('amount'),
            'pending_amount' => $invoices->where('status', 'pending')->sum('amount'),
            'success_rate' => $invoices->count() > 0 ? round(($invoices->where('status', 'paid')->count() / $invoices->count()) * 100, 2) : 0,
        ];

        // Monthly data for charts (last 12 months)
        $monthlyDataArray = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthInvoices = $invoices->whereBetween('created_at', [
                $date->startOfMonth(),
                $date->endOfMonth()
            ]);

            $monthlyDataArray[] = [
                'month' => $date->format('M Y'),
                'total' => $monthInvoices->count(),
                'paid' => $monthInvoices->where('status', 'paid')->count(),
                'amount' => $monthInvoices->where('status', 'paid')->sum('amount'),
            ];
        }

        $perPage = 6;
        $currentPage = \Illuminate\Pagination\Paginator::resolveCurrentPage('page');
        $currentItems = collect($monthlyDataArray)->forPage($currentPage, $perPage);
        $monthlyData = new \Illuminate\Pagination\LengthAwarePaginator(
            $currentItems,
            count($monthlyDataArray),
            $perPage,
            $currentPage,
            ['path' => request()->url(), 'pageName' => 'page']
        );

        return view('dashboard.analytics', compact('analytics', 'monthlyData', 'monthlyDataArray'));
    }

    public function analyticsApi(Request $request)
    {
        $invoices = Invoice::all();

        // Analytics data
        $analytics = [
            'total_invoices' => $invoices->count(),
            'paid_invoices' => $invoices->where('status', 'paid')->count(),
            'pending_invoices' => $invoices->where('status', 'pending')->count(),
            'cancelled_invoices' => $invoices->where('status', 'cancelled')->count(),
            'total_amount' => $invoices->sum('amount'),
            'paid_amount' => $invoices->where('status', 'paid')->sum('amount'),
            'pending_amount' => $invoices->where('status', 'pending')->sum('amount'),
            'success_rate' => $invoices->count() > 0 ? round(($invoices->where('status', 'paid')->count() / $invoices->count()) * 100, 2) : 0,
        ];

        // Monthly data for charts (last 12 months)
        $monthlyDataArray = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthInvoices = $invoices->whereBetween('created_at', [
                $date->startOfMonth(),
                $date->endOfMonth()
            ]);

            $monthlyDataArray[] = [
                'month' => $date->format('M Y'),
                'total' => $monthInvoices->count(),
                'paid' => $monthInvoices->where('status', 'paid')->count(),
                'amount' => $monthInvoices->where('status', 'paid')->sum('amount'),
            ];
        }

        return response()->json([
            'analytics' => $analytics,
            'monthlyData' => $monthlyDataArray,
        ]);
    }
}
