<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    public function index()
    {
        $payments = Payment::with('invoice')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $stats = [
            'total_payments' => Payment::count(),
            'successful_payments' => Payment::where('status', 'successful')->count(),
            'failed_payments' => Payment::where('status', 'failed')->count(),
            'total_amount' => Payment::where('status', 'successful')->sum('amount'),
        ];

        return view('payments.index', compact('payments', 'stats'));
    }

    public function pay(Request $request, Invoice $invoice, $token)
    {
        // Validate token
        if ($invoice->token !== $token || $invoice->isExpired() || $invoice->isPaid()) {
            abort(404);
        }

        return view('payments.pay', compact('invoice'));
    }

    public function initiateStkPush(Request $request, Invoice $invoice)
    {
        $request->validate([
            'phone_number' => 'required|regex:/^254[0-9]{9}$/',
        ]);

        // Check if invoice is still valid
        if ($invoice->isExpired() || $invoice->isPaid()) {
            return response()->json(['error' => 'Invoice is no longer valid'], 400);
        }

        // Create payment record
        $payment = Payment::create([
            'invoice_id' => $invoice->id,
            'phone_number' => $request->phone_number,
            'amount' => $invoice->amount,
        ]);

        // Initiate STK Push
        $response = $this->initiateMpesaStkPush($payment);

        if ($response['success']) {
            return response()->json([
                'success' => true,
                'message' => 'STK Push initiated successfully. Please check your phone for the M-Pesa prompt.',
                'payment_id' => $payment->id
            ]);
        } else {
            $payment->update(['status' => 'failed']);
            return response()->json(['error' => $response['message']], 400);
        }
    }

    public function callback(Request $request)
    {
        Log::info('M-Pesa Callback Received', $request->all());

        $data = $request->all();

        if (isset($data['Body']['stkCallback'])) {
            $callbackData = $data['Body']['stkCallback'];
            $merchantRequestId = $callbackData['MerchantRequestID'];
            $resultCode = $callbackData['ResultCode'];

            // Find payment by merchant request ID
            $payment = Payment::where('merchant_request_id', $merchantRequestId)->first();

            if ($payment) {
                if ($resultCode == 0) {
                    // Payment successful
                    $metadata = $callbackData['CallbackMetadata']['Item'] ?? [];
                    $updateData = [
                        'status' => 'successful',
                        'paid_at' => now(), // Default to now if TransactionDate not provided
                        'callback_data' => $callbackData,
                    ];

                    foreach ($metadata as $item) {
                        switch ($item['Name']) {
                            case 'Amount':
                                $updateData['amount'] = $item['Value'];
                                break;
                            case 'M-Pesa Receipt Number':
                                $updateData['receipt_number'] = $item['Value'];
                                $updateData['transaction_id'] = $item['Value'];
                                break;
                            case 'TransactionDate':
                                // Parse TransactionDate (format: YYYYMMDDHHMMSS)
                                try {
                                    $updateData['paid_at'] = \Carbon\Carbon::createFromFormat('YmdHis', $item['Value']);
                                } catch (\Exception $e) {
                                    Log::warning('Failed to parse TransactionDate', ['date' => $item['Value'], 'error' => $e->getMessage()]);
                                    // Keep default paid_at as now()
                                }
                                break;
                            case 'PhoneNumber':
                                $updateData['phone_number'] = $item['Value'];
                                break;
                        }
                    }

                    $payment->update($updateData);

                    // Mark invoice as paid and expired
                    $payment->invoice->markAsPaid();
                    $payment->invoice->markAsExpired();
                } else {
                    // Payment failed
                    $payment->update([
                        'status' => 'failed',
                        'callback_data' => $callbackData,
                    ]);
                }
            } else {
                Log::error('Payment not found for MerchantRequestID', ['merchant_request_id' => $merchantRequestId]);
            }
        }

        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);
    }

    public function queryStkPush(Request $request, Payment $payment)
    {
        $response = $this->queryStkPushStatus($payment);

        if ($response['success']) {
            return response()->json(['success' => true, 'data' => $response['data']]);
        } else {
            return response()->json(['error' => $response['message']], 400);
        }
    }

    public function queryTransactionStatus(Request $request, Payment $payment)
    {
        $response = $this->queryTransactionStatusAPI($payment);

        if ($response['success']) {
            return response()->json(['success' => true, 'data' => $response['data']]);
        } else {
            return response()->json(['error' => $response['message']], 400);
        }
    }

    public function reverseTransaction(Request $request, Payment $payment)
    {
        $request->validate([
            'reason' => 'required|string|max:255',
        ]);

        if (!$payment->isSuccessful()) {
            return response()->json(['error' => 'Only successful transactions can be reversed'], 400);
        }

        $response = $this->reverseMpesaTransaction($payment, $request->reason);

        if ($response['success']) {
            $payment->update(['status' => 'reversed']);
            return response()->json(['success' => true, 'message' => 'Transaction reversed successfully']);
        } else {
            return response()->json(['error' => $response['message']], 400);
        }
    }

    private function initiateMpesaStkPush(Payment $payment)
    {
        // M-Pesa Daraja API credentials (store in config or .env)
        $consumerKey = config('mpesa.consumer_key');
        $consumerSecret = config('mpesa.consumer_secret');
        $shortcode = config('mpesa.shortcode');
        $passkey = config('mpesa.passkey');
        $baseUrl = config('mpesa.base_url');

        // Get access token
        $accessTokenResponse = Http::withBasicAuth($consumerKey, $consumerSecret)
            ->get($baseUrl . '/oauth/v1/generate?grant_type=client_credentials');

        if (!$accessTokenResponse->successful()) {
            Log::error('Failed to get M-Pesa access token', ['response' => $accessTokenResponse->body()]);
            return ['success' => false, 'message' => 'Failed to get access token'];
        }

        $accessToken = $accessTokenResponse->json()['access_token'];

        // Prepare STK Push request
        $timestamp = now()->format('YmdHis');
        $password = base64_encode($shortcode . $passkey . $timestamp);

        $stkPushData = [
            'BusinessShortCode' => $shortcode,
            'Password' => $password,
            'Timestamp' => $timestamp,
            'TransactionType' => 'CustomerPayBillOnline',
            'Amount' => (int) round($payment->amount), // Round and cast to integer for M-Pesa
            'PartyA' => $payment->phone_number,
            'PartyB' => '542542',
            'PhoneNumber' => $payment->phone_number,
            'CallBackURL' => 'https://acorns.sspnetworks.co.ke/payments/callback',
            'AccountReference' => '844811',
            'TransactionDesc' => 'Payment for ' . $payment->invoice->description,
        ];

        Log::info('Initiating STK Push', ['data' => $stkPushData]);

        $response = Http::withToken($accessToken)
            ->post($baseUrl . '/mpesa/stkpush/v1/processrequest', $stkPushData);

        Log::info('STK Push API Response', ['status' => $response->status(), 'body' => $response->body()]);

        if ($response->successful()) {
            $data = $response->json();
            if (isset($data['ResponseCode']) && $data['ResponseCode'] == '0') {
                // Store merchant request ID for callback matching
                $payment->update(['merchant_request_id' => $data['MerchantRequestID']]);
                return ['success' => true, 'message' => 'STK Push initiated successfully'];
            } else {
                return ['success' => false, 'message' => $data['CustomerMessage'] ?? 'STK Push failed'];
            }
        }

        return ['success' => false, 'message' => 'Failed to initiate STK Push'];
    }

    private function queryStkPushStatus(Payment $payment)
    {
        $consumerKey = config('mpesa.consumer_key');
        $consumerSecret = config('mpesa.consumer_secret');
        $shortcode = config('mpesa.shortcode');
        $passkey = config('mpesa.passkey');
        $baseUrl = config('mpesa.base_url');

        // Get access token
        $accessTokenResponse = Http::withBasicAuth($consumerKey, $consumerSecret)
            ->get($baseUrl . '/oauth/v1/generate?grant_type=client_credentials');

        if (!$accessTokenResponse->successful()) {
            return ['success' => false, 'message' => 'Failed to get access token'];
        }

        $accessToken = $accessTokenResponse->json()['access_token'];

        $timestamp = now()->format('YmdHis');
        $password = base64_encode($shortcode . $passkey . $timestamp);

        $queryData = [
            'BusinessShortCode' => $shortcode,
            'Password' => $password,
            'Timestamp' => $timestamp,
            'CheckoutRequestID' => $payment->merchant_request_id,
        ];

        $response = Http::withToken($accessToken)
            ->post($baseUrl . '/mpesa/stkpushquery/v1/query', $queryData);

        if ($response->successful()) {
            $data = $response->json();
            return ['success' => true, 'data' => $data];
        }

        return ['success' => false, 'message' => 'Failed to query STK Push status'];
    }

    private function queryTransactionStatusAPI(Payment $payment)
    {
        $consumerKey = config('mpesa.consumer_key');
        $consumerSecret = config('mpesa.consumer_secret');
        $shortcode = config('mpesa.shortcode');
        $passcode = config('mpesa.passkey');
        $baseUrl = config('mpesa.base_url');

        // Get access token
        $accessTokenResponse = Http::withBasicAuth($consumerKey, $consumerSecret)
            ->get($baseUrl . '/oauth/v1/generate?grant_type=client_credentials');

        if (!$accessTokenResponse->successful()) {
            return ['success' => false, 'message' => 'Failed to get access token'];
        }

        $accessToken = $accessTokenResponse->json()['access_token'];

        $timestamp = now()->format('YmdHis');
        $password = base64_encode($shortcode . $passcode . $timestamp);

        $queryData = [
            'Initiator' => config('mpesa.initiator_name', 'testapi'),
            'SecurityCredential' => config('mpesa.security_credential', ''),
            'CommandID' => 'TransactionStatusQuery',
            'TransactionID' => $payment->transaction_id,
            'PartyA' => $shortcode,
            'IdentifierType' => '4', // MSISDN
            'ResultURL' => 'https://acorns.sspnetworks.co.ke/payments/callback',
            'QueueTimeOutURL' => 'https://acorns.sspnetworks.co.ke/payments/callback',
            'Remarks' => 'Transaction status query',
            'Occasion' => 'Transaction status check',
        ];

        $response = Http::withToken($accessToken)
            ->post($baseUrl . '/mpesa/transactionstatus/v1/query', $queryData);

        if ($response->successful()) {
            $data = $response->json();
            return ['success' => true, 'data' => $data];
        }

        return ['success' => false, 'message' => 'Failed to query transaction status'];
    }

    private function reverseMpesaTransaction(Payment $payment, $reason)
    {
        $consumerKey = config('mpesa.consumer_key');
        $consumerSecret = config('mpesa.consumer_secret');
        $shortcode = config('mpesa.shortcode');
        $passcode = config('mpesa.passkey');
        $baseUrl = config('mpesa.base_url');

        // Get access token
        $accessTokenResponse = Http::withBasicAuth($consumerKey, $consumerSecret)
            ->get($baseUrl . '/oauth/v1/generate?grant_type=client_credentials');

        if (!$accessTokenResponse->successful()) {
            return ['success' => false, 'message' => 'Failed to get access token'];
        }

        $accessToken = $accessTokenResponse->json()['access_token'];

        $timestamp = now()->format('YmdHis');
        $password = base64_encode($shortcode . $passcode . $timestamp);

        $reversalData = [
            'Initiator' => config('mpesa.initiator_name', 'testapi'),
            'SecurityCredential' => config('mpesa.security_credential', ''),
            'CommandID' => 'TransactionReversal',
            'TransactionID' => $payment->transaction_id,
            'Amount' => $payment->amount,
            'ReceiverParty' => $payment->phone_number,
            'RecieverIdentifierType' => '4', // MSISDN
            'ResultURL' => 'https://acorns.sspnetworks.co.ke/payments/callback',
            'QueueTimeOutURL' => 'https://acorns.sspnetworks.co.ke/payments/callback',
            'Remarks' => $reason,
            'Occasion' => 'Transaction reversal',
        ];

        $response = Http::withToken($accessToken)
            ->post($baseUrl . '/mpesa/reversal/v1/request', $reversalData);

        if ($response->successful()) {
            $data = $response->json();
            if (isset($data['ResponseCode']) && $data['ResponseCode'] == '0') {
                return ['success' => true, 'data' => $data];
            }
        }

        return ['success' => false, 'message' => 'Failed to reverse transaction'];
    }

    public function getPaymentStatus(Payment $payment)
    {
        return response()->json([
            'status' => $payment->status,
            'payment_id' => $payment->id,
            'transaction_id' => $payment->transaction_id,
            'receipt_number' => $payment->receipt_number
        ]);
    }

    public function showSuccess(Payment $payment)
    {
        return view('payments.success', compact('payment'));
    }
}
