<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Complete Your Payment</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .btn-success {
            background-color: #28a745;
            border-color: #28a745;
        }
        .btn-success:hover {
            background-color: #218838;
            border-color: #1e7e34;
        }
    </style>
</head>
<body>
<div class="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 py-12">
    <div class="max-w-2xl mx-auto px-4">
        <!-- Invoice Summary Card -->
        <div class="bg-white rounded-2xl shadow-xl border border-gray-200 overflow-hidden card-hover mb-8">
            <div class="bg-gradient-to-r from-green-600 to-emerald-600 px-8 py-6">
                <div class="flex items-center">
                    <div class="bg-white bg-opacity-20 p-3 rounded-xl">
                        <i class="fas fa-mobile-alt text-white text-2xl"></i>
                    </div>
                    <div class="ml-4">
                        <h1 class="text-2xl font-bold text-white">Complete Your Payment</h1>
                        <p class="text-green-100 mt-1">Secure M-Pesa payment for your invoice</p>
                    </div>
                </div>
            </div>

            <div class="p-8">
                <div class="text-center mb-8">
                    <div class="inline-flex items-center justify-center w-20 h-20 bg-green-100 rounded-full mb-4">
                        <i class="fas fa-building text-3xl text-green-600"></i>
                    </div>
                    <h2 class="text-2xl font-bold text-gray-900 mb-2">{{ $invoice->business_name }}</h2>
                    <p class="text-gray-600 mb-4">{{ $invoice->description }}</p>
                    <div class="bg-gray-50 rounded-xl p-6 inline-block">
                        <p class="text-sm text-gray-600 mb-1">Amount Due</p>
                        <p class="text-4xl font-bold text-green-600">Ksh {{ number_format($invoice->amount, 2) }}</p>
                    </div>
                </div>

                <!-- Invoice Details -->
                <div class="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-8">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                        <i class="fas fa-info-circle mr-2 text-blue-600"></i>Invoice Details
                    </h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                        <div>
                            <p class="text-gray-600">Invoice Number</p>
                            <p class="font-semibold text-gray-900">{{ $invoice->invoice_number }}</p>
                        </div>
                        <div>
                            <p class="text-gray-600">Customer</p>
                            <p class="font-semibold text-gray-900">{{ $invoice->customer_name }}</p>
                        </div>
                        @if($invoice->due_date)
                        <div>
                            <p class="text-gray-600">Due Date</p>
                            <p class="font-semibold text-gray-900">{{ $invoice->due_date->format('M d, Y') }}</p>
                        </div>
                        @endif
                        <div>
                            <p class="text-gray-600">Status</p>
                            <span class="px-2 py-1 text-xs font-semibold rounded-full bg-yellow-100 text-yellow-800">
                                {{ ucfirst($invoice->status) }}
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Payment Form -->
                <form id="payment-form" method="POST" action="{{ route('payments.stk-push', $invoice) }}" class="space-y-6">
                    @csrf

                    <div>
                        <label for="phone_number" class="block text-sm font-semibold text-gray-700 mb-2 flex items-center">
                            <i class="fas fa-phone mr-2 text-green-600"></i>M-Pesa Phone Number
                        </label>
                        <div class="relative">
                            <input type="tel" name="phone_number" id="phone_number" placeholder="254712345678" required
                                   class="w-full pl-4 pr-4 py-4 text-lg border border-gray-300 rounded-xl shadow-sm focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-green-500 transition-all duration-200"
                                   pattern="254[0-9]{9}" title="Please enter a valid Kenyan phone number starting with 254">
                            <div class="absolute right-4 top-4 text-gray-400">
                                <i class="fas fa-mobile-alt"></i>
                            </div>
                        </div>
                        <p class="mt-2 text-sm text-gray-600 flex items-center">
                            <i class="fas fa-info-circle mr-1"></i>
                            Enter your M-Pesa registered phone number (e.g., 254712345678)
                        </p>
                        @error('phone_number')
                            <p class="mt-2 text-sm text-red-600 flex items-center"><i class="fas fa-exclamation-circle mr-1"></i>{{ $message }}</p>
                        @enderror
                    </div>

                    <button type="submit" id="pay-button" class="w-full btn-success text-white px-8 py-4 rounded-xl hover:shadow-xl flex items-center justify-center font-semibold text-lg transition-all duration-200">
                        <i class="fas fa-credit-card mr-3"></i>Proceed to M-Pesa Payment
                    </button>
                </form>

                <!-- Status Messages -->
                <div id="status-messages" class="mt-6 hidden">
                    <div id="stk-success-message" class="bg-green-50 border border-green-200 rounded-xl p-4 hidden">
                        <div class="flex items-center">
                            <i class="fas fa-check-circle text-green-600 mr-3"></i>
                            <div>
                                <p class="text-sm font-semibold text-green-800">STK Push Sent!</p>
                                <p class="text-sm text-green-700">Please check your phone for the M-Pesa payment prompt.</p>
                            </div>
                        </div>
                    </div>

                    <div id="payment-success-message" class="bg-green-50 border border-green-200 rounded-xl p-4 hidden">
                        <div class="flex items-center">
                            <i class="fas fa-check-circle text-green-600 mr-3"></i>
                            <div>
                                <p class="text-sm font-semibold text-green-800">Payment Successful!</p>
                                <p class="text-sm text-green-700">Your payment has been processed successfully.</p>
                                <p id="receipt-info" class="text-sm text-green-600 mt-2 font-mono"></p>
                                <div class="mt-4">
                                    <a href="#" id="view-receipt-btn" class="inline-flex items-center px-4 py-2 bg-green-600 text-white text-sm font-semibold rounded-lg hover:bg-green-700 transition-colors">
                                        <i class="fas fa-eye mr-2"></i>View Receipt
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div id="error-message" class="bg-red-50 border border-red-200 rounded-xl p-4 hidden">
                        <div class="flex items-center">
                            <i class="fas fa-exclamation-circle text-red-600 mr-3"></i>
                            <div>
                                <p class="text-sm font-semibold text-red-800">Payment Failed</p>
                                <p id="error-text" class="text-sm text-red-700"></p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Instructions -->
                <div class="mt-8 bg-yellow-50 border border-yellow-200 rounded-xl p-6">
                    <h4 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                        <i class="fas fa-lightbulb mr-2 text-yellow-600"></i>How it works
                    </h4>
                    <div class="space-y-3 text-sm text-gray-700">
                        <div class="flex items-start">
                            <span class="bg-yellow-200 text-yellow-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-bold mr-3 mt-0.5">1</span>
                            <p>Click "Proceed to M-Pesa Payment" to initiate the transaction</p>
                        </div>
                        <div class="flex items-start">
                            <span class="bg-yellow-200 text-yellow-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-bold mr-3 mt-0.5">2</span>
                            <p>You'll receive an M-Pesa prompt on your phone</p>
                        </div>
                        <div class="flex items-start">
                            <span class="bg-yellow-200 text-yellow-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-bold mr-3 mt-0.5">3</span>
                            <p>Enter your M-Pesa PIN to complete the payment</p>
                        </div>
                        <div class="flex items-start">
                            <span class="bg-yellow-200 text-yellow-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-bold mr-3 mt-0.5">4</span>
                            <p>You'll be redirected to a success page once payment is confirmed</p>
                        </div>
                    </div>
                </div>

                <!-- Security Notice -->
                <div class="mt-6 bg-green-50 border border-green-200 rounded-xl p-4">
                    <div class="flex items-center">
                        <i class="fas fa-shield-alt text-green-600 mr-3"></i>
                        <div>
                            <p class="text-sm font-semibold text-green-800">Secure Payment</p>
                            <p class="text-sm text-green-700">Your payment is processed securely through M-Pesa's trusted network</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<script>
document.getElementById('payment-form').addEventListener('submit', function(e) {
    e.preventDefault();

    const form = this;
    const button = document.getElementById('pay-button');
    const statusMessages = document.getElementById('status-messages');
    const stkSuccessMessage = document.getElementById('stk-success-message');
    const paymentSuccessMessage = document.getElementById('payment-success-message');
    const errorMessage = document.getElementById('error-message');
    const errorText = document.getElementById('error-text');

    // Hide all messages initially
    statusMessages.classList.add('hidden');
    stkSuccessMessage.classList.add('hidden');
    paymentSuccessMessage.classList.add('hidden');
    errorMessage.classList.add('hidden');

    // Disable button and show loading
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin mr-3"></i>Processing...';

    // Create FormData
    const formData = new FormData(form);

    // Send AJAX request
    fetch(form.action, {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '{{ csrf_token() }}',
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Show STK success message immediately
            statusMessages.classList.remove('hidden');
            stkSuccessMessage.classList.remove('hidden');

            // Hide form
            form.style.display = 'none';

            // Start polling for payment status
            const paymentId = data.payment_id;
            pollPaymentStatus(paymentId);
        } else {
            // Show error message
            statusMessages.classList.remove('hidden');
            errorMessage.classList.remove('hidden');
            errorText.textContent = data.error || 'An error occurred';

            // Re-enable button
            button.disabled = false;
            button.innerHTML = '<i class="fas fa-credit-card mr-3"></i>Proceed to M-Pesa Payment';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        // Show error message
        statusMessages.classList.remove('hidden');
        errorMessage.classList.remove('hidden');
        errorText.textContent = 'Network error. Please try again.';

        // Re-enable button
        button.disabled = false;
        button.innerHTML = '<i class="fas fa-credit-card mr-3"></i>Proceed to M-Pesa Payment';
    });
});

function pollPaymentStatus(paymentId) {
    const checkStatus = () => {
        fetch(`/payments/${paymentId}/status`, {
            method: 'GET',
            headers: {
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'successful') {
                // Show payment success message with receipt number
                document.getElementById('stk-success-message').classList.add('hidden');
                document.getElementById('payment-success-message').classList.remove('hidden');
                if (data.receipt_number) {
                    document.getElementById('receipt-info').textContent = `M-Pesa Receipt: ${data.receipt_number}`;
                    // Update the view receipt button to redirect to success page
                    document.getElementById('view-receipt-btn').href = `/payments/${paymentId}/success`;
                }
            } else if (data.status === 'failed') {
                // Show error message
                document.getElementById('stk-success-message').classList.add('hidden');
                document.getElementById('error-message').classList.remove('hidden');
                document.getElementById('error-text').textContent = 'Payment was not completed successfully.';
            } else {
                // Continue polling
                setTimeout(checkStatus, 3000); // Check every 3 seconds
            }
        })
        .catch(error => {
            console.error('Error checking status:', error);
            setTimeout(checkStatus, 3000);
        });
    };

    // Start polling
    setTimeout(checkStatus, 5000); // Start checking after 5 seconds
}
</script>
</body>
</html>
