<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Payment;
use App\Models\Invoice;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Log;

class PaymentCallbackTest extends TestCase
{
    use RefreshDatabase;

    public function test_successful_mpesa_callback_updates_payment()
    {
        // Create an invoice
        $invoice = Invoice::factory()->create();

        // Create a payment with merchant_request_id
        $payment = Payment::create([
            'invoice_id' => $invoice->id,
            'merchant_request_id' => '12345-67890-1234',
            'phone_number' => '254712345678',
            'amount' => 100.00,
            'status' => 'pending',
        ]);

        // Simulate successful callback data
        $callbackData = [
            'Body' => [
                'stkCallback' => [
                    'MerchantRequestID' => '12345-67890-1234',
                    'ResultCode' => 0,
                    'ResultDesc' => 'The service request is processed successfully.',
                    'CallbackMetadata' => [
                        'Item' => [
                            [
                                'Name' => 'Amount',
                                'Value' => 100.00
                            ],
                            [
                                'Name' => 'M-Pesa Receipt Number',
                                'Value' => 'ABC123XYZ'
                            ],
                            [
                                'Name' => 'TransactionDate',
                                'Value' => '20231101120000' // YYYYMMDDHHMMSS
                            ],
                            [
                                'Name' => 'PhoneNumber',
                                'Value' => '254712345678'
                            ]
                        ]
                    ]
                ]
            ]
        ];

        // Send POST request to callback endpoint
        $response = $this->postJson('/payments/callback', $callbackData);

        $response->assertStatus(200)
                 ->assertJson(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);

        // Refresh payment from database
        $payment->refresh();

        // Assert payment fields are updated
        $this->assertEquals('successful', $payment->status);
        $this->assertEquals('ABC123XYZ', $payment->transaction_id);
        $this->assertEquals('ABC123XYZ', $payment->receipt_number);
        $this->assertEquals(100.00, $payment->amount);
        $this->assertEquals('254712345678', $payment->phone_number);
        $this->assertNotNull($payment->paid_at);
        $this->assertEquals($callbackData['Body']['stkCallback'], $payment->callback_data);

        // Assert invoice is marked as paid and expired
        $invoice->refresh();
        $this->assertTrue($invoice->is_paid);
        $this->assertTrue($invoice->is_expired);
    }

    public function test_failed_mpesa_callback_updates_payment()
    {
        // Create an invoice
        $invoice = Invoice::factory()->create();

        // Create a payment with merchant_request_id
        $payment = Payment::create([
            'invoice_id' => $invoice->id,
            'merchant_request_id' => '12345-67890-1235',
            'phone_number' => '254712345678',
            'amount' => 100.00,
            'status' => 'pending',
        ]);

        // Simulate failed callback data
        $callbackData = [
            'Body' => [
                'stkCallback' => [
                    'MerchantRequestID' => '12345-67890-1235',
                    'ResultCode' => 1,
                    'ResultDesc' => 'The service request failed.',
                    'CallbackMetadata' => [
                        'Item' => []
                    ]
                ]
            ]
        ];

        // Send POST request to callback endpoint
        $response = $this->postJson('/payments/callback', $callbackData);

        $response->assertStatus(200)
                 ->assertJson(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);

        // Refresh payment from database
        $payment->refresh();

        // Assert payment status is updated to failed
        $this->assertEquals('failed', $payment->status);
        $this->assertEquals($callbackData['Body']['stkCallback'], $payment->callback_data);

        // Assert other fields are not updated
        $this->assertNull($payment->transaction_id);
        $this->assertNull($payment->receipt_number);
        $this->assertEquals(100.00, $payment->amount); // Should remain as is
        $this->assertEquals('254712345678', $payment->phone_number); // Should remain as is
        $this->assertNull($payment->paid_at);
    }

    public function test_callback_with_missing_metadata()
    {
        // Create an invoice
        $invoice = Invoice::factory()->create();

        // Create a payment with merchant_request_id
        $payment = Payment::create([
            'invoice_id' => $invoice->id,
            'merchant_request_id' => '12345-67890-1236',
            'phone_number' => '254712345678',
            'amount' => 100.00,
            'status' => 'pending',
        ]);

        // Simulate callback with partial metadata
        $callbackData = [
            'Body' => [
                'stkCallback' => [
                    'MerchantRequestID' => '12345-67890-1236',
                    'ResultCode' => 0,
                    'ResultDesc' => 'The service request is processed successfully.',
                    'CallbackMetadata' => [
                        'Item' => [
                            [
                                'Name' => 'M-Pesa Receipt Number',
                                'Value' => 'XYZ789ABC'
                            ]
                            // Missing Amount, TransactionDate, PhoneNumber
                        ]
                    ]
                ]
            ]
        ];

        // Send POST request to callback endpoint
        $response = $this->postJson('/payments/callback', $callbackData);

        $response->assertStatus(200);

        // Refresh payment from database
        $payment->refresh();

        // Assert only available fields are updated
        $this->assertEquals('successful', $payment->status);
        $this->assertEquals('XYZ789ABC', $payment->transaction_id);
        $this->assertEquals('XYZ789ABC', $payment->receipt_number);
        // Amount, phone_number, paid_at should remain as initial or default
        $this->assertEquals(100.00, $payment->amount);
        $this->assertEquals('254712345678', $payment->phone_number);
        $this->assertNotNull($payment->paid_at); // Should be set to now() since TransactionDate missing
    }

    public function test_callback_for_nonexistent_payment_logs_error()
    {
        // Simulate callback for non-existent merchant_request_id
        $callbackData = [
            'Body' => [
                'stkCallback' => [
                    'MerchantRequestID' => 'nonexistent-123',
                    'ResultCode' => 0,
                    'ResultDesc' => 'Success',
                    'CallbackMetadata' => [
                        'Item' => []
                    ]
                ]
            ]
        ];

        // Send POST request to callback endpoint
        $response = $this->postJson('/payments/callback', $callbackData);

        $response->assertStatus(200);

        // Assert no payment is updated (since none exists)
        $this->assertDatabaseMissing('payments', ['merchant_request_id' => 'nonexistent-123']);
    }
}
